/*
 *  impute.c
 *  PenLSCens
 *
 *  Created by Brent Johnson on 3/11/08.
 *  Copyright 2008 __MyCompanyName__. All rights reserved.
 *
 */

/* ystar = output
 * n = sample size
 * ei = logU - mu, i=1,...,n
 * mu = linear predictor, i=1,...,n
 * Surv(t) = from t=0 to t=\infty
 * UniqRes = support of S(t)
 * nuniq = length of UniqRes, Surv
 */
 
#include <R.h>
#include <Rmath.h>

double impute_integral(double e1, double * Surv, double * UniqRes, int nuniq);
double InvertSurvAtEi(double e1, double * UniqRes, double * Surv, int nuniq);

void impute(double * ystar, int * n, double * ei, double * del, double * mu, double * Surv, 
	double * UniqRes, int * nuniq)
{
	int i ;
    for (i=0;i<*n;i++)
    {
		
        if (*(del+i) < 0.5)
        {
            double ThisEi = *(ei+i) ;
            double MyInt = impute_integral(ThisEi, Surv, UniqRes, *nuniq) ;
            double Fbar = InvertSurvAtEi(ThisEi, UniqRes, Surv, *nuniq) ;
            double ehat = ThisEi + MyInt/Fbar ;

			*(ystar+i) = ehat + *(mu+i);
        }
		else 
		{
			*(ystar+i) = *(ei+i) + *(mu+i);
		}
	}
}	

double InvertSurvAtEi(double e1, double * UniqRes, double * Surv, int nuniq)
{
    double val = 1.0 ;
    int uniq_cntr = 0 ;
    double * s_ptr = Surv ;
    double * e_ptr = UniqRes ;
    
    if (e1 > *(UniqRes+nuniq-1)) // that is, greater than the last UniqRes
    {
        val = 0.0 ; // special for this Buckley-James software
        //val = *(Surv+nuniq-1) ; // this is what is typically done
    }
    else
    {
        do
        {
            uniq_cntr++ ;
            if (e1 <= *e_ptr) 
            {
                val = *s_ptr;
                break ;
            }
            ++s_ptr;
            ++e_ptr;
        } while (uniq_cntr < nuniq) ;
    }
    
    return val ;
}


double impute_integral(double e1, double * Surv, double * UniqRes, int nuniq)
{
    double MyInt = 0.0 ;
    double * res_ptr = UniqRes + nuniq - 1;
    double * surv_ptr = Surv + nuniq - 1;
    
    if (e1 < *res_ptr) // that is, integral = 0.0 for observations past last UniqRes
    {
        int uniq_cntr = 0 ;
        double Ej=*res_ptr;
        double Sj=*surv_ptr;
        --res_ptr;
        --surv_ptr;
        //uniq_cntr++ ;
        do
        {
            uniq_cntr++ ;
            //printf("\t uniq_cntr = %d out of %d\n", uniq_cntr, nuniq) ;
            
            if (e1>*res_ptr) // this clause catches subjects who were censored in between failures
            {
                MyInt += Sj * (Ej - e1) ;
                //printf("ei = %6.4f : e(j-1)=%6.4f S(j-1)=%6.4f MyInt = %6.4f\n", e1, Ej, Sj, MyInt) ;
                break ;
            }
            
            MyInt += Sj * (Ej - *res_ptr) ;
            //printf("ei = %6.4f : e(j-1)=%6.4f e(j)=%6.4f S(j)=%6.4f MyInt = %6.4f\n", e1, *res_ptr, Ej, Sj, MyInt) ;
            Ej=*res_ptr ;
            Sj=*surv_ptr ;
            
            if (uniq_cntr==(nuniq-1)) // this clause catches observations before the first failure
            {
                //printf("cntr = nuniq-1\n") ;
                MyInt += Sj * (Ej - e1) ; // should be, [1.0 * (Ej - e1)]
                //printf("ei = %6.4f : ei = %6.4f e(0)=%6.4f S(0)=%6.4f MyInt = %6.4f\n", e1, e1, Ej, Sj, MyInt) ;
                
                break ; // catch before you point to something incorrect
            }
            --res_ptr ;
            --surv_ptr ;
            
        } while (uniq_cntr < nuniq);
        //} while (uniq_cntr < (nuniq-1));
        
        //printf("ei = %6.4f e(j)=%6.4f e(j-1)=%6.4f S(j)=%6.4f MyInt = %6.4f\n", e1, Ej, *res_ptr, *surv_ptr, MyInt) ;
        
    }
    
    return MyInt ;
}

